'''Examples.py'''

kwarg = lambda name,dflt,typ,args: typ(args[name]) if name in args else dflt

typename = lambda obj: type(obj).__name__


class dualnumber:
    Etype = 'Sorry. Unknown type: {}'

    def __init__ (self, real=0, dual=0):
        '''New Dual Number instance.'''
        self.real = real
        self.dual = dual

    def __repr__ (self):
        '''Return debug string.'''
        return f'<{typename(self)} @{id(self):08x}>'

    def __str__ (self):
        '''Return printable string.'''
        return f'({self.real:+}{self.dual:+}e)'

    @staticmethod
    def differentiate (function, x):
        '''Differentiate function at x and return value.'''
        return function(dualnumber(x,1)).dual

    def __add__  (self, other):
        '''Add with a DN or a numeric value and return new DN.'''

        # First see if other is another dualnumber...
        if isinstance(other,dualnumber):
            # Do the math...
            ta = self.real + other.real
            tb = self.dual + other.dual
            # Return a new dual number...
            return dualnumber(ta, tb)

        # Alternately, handle integers and floats...
        if isinstance(other,int) or isinstance(other,float):
            # The math here involves just the real part...
            ta = self.real + other
            tb = self.dual
            # Return a new dual number...
            return dualnumber(ta, tb)

        # If none of the above, that's an error...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    # Addition is communitive, so Right-Add is same as Left-Add...
    __radd__ = __add__

    def __iadd__  (self, other):
        '''Add a DN or numeric value to self.'''

        # First, deal with another dualnumber...
        if isinstance(other,dualnumber):
            # Do the math to self...
            self.real += other.real
            self.dual += other.dual
            return self

        # Alternately, integers or floats...
        if isinstance(other,int) or isinstance(other,float):
            self.real += other
            return self

        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __sub__  (self, other):
        '''Subtract with a DN or a numeric value and return new DN.'''
        if isinstance(other,dualnumber):
            ta = self.real - other.real
            tb = self.dual - other.dual
            return dualnumber(ta, tb)
        if isinstance(other,int) or isinstance(other,float):
            ta = self.real - other
            tb = self.dual
            return dualnumber(ta, tb)
        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __rsub__ (self, other):
        '''Subtract with a DN or a numeric value and return new DN.
           Note that subtraction does NOT commute so we must implement rsub.
           Note also that other will never be a dualnumber, so we only need
           to handle ints and floats.'''
        if isinstance(other,int) or isinstance(other,float):
            ta = other - self.real
            tb = self.dual
            return dualnumber(ta, tb)
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __isub__  (self, other):
        '''Subtract a DN or numeric value from self.'''
        if isinstance(other,dualnumber):
            self.real -= other.real
            self.dual -= other.dual
            return self
        if isinstance(other,int) or isinstance(other,float):
            self.real -= other
            return self
        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __mul__  (self, other):
        '''Multiply by a DN or a numeric value and return new DN.'''
        
        # The multiplication math is a bit more involved...
        if isinstance(other,dualnumber):
            ta = self.real * other.real
            tb = (self.real * other.dual) + (self.dual * other.real)
            return dualnumber(ta, tb)

        # Scalar values apply to both real and dual parts...
        if isinstance(other,int) or isinstance(other,float):
            ta = self.real * other
            tb = self.dual * other
            return dualnumber(ta, tb)
        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    # Right-Mul same as Left-Mul...
    __rmul__ = __mul__

    def __imul__  (self, other):
        '''Multiply a DN or numeric value to self.'''
        if isinstance(other,dualnumber):
            ta = self.real * other.real
            tb = (self.real * other.dual) + (self.dual * other.real)
            self.real = ta
            self.dual = tb
            return self
        if isinstance(other,int) or isinstance(other,float):
            self.real *= other
            self.dual *= other
            return self
        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __pow__ (self, other, modulo=None):
        '''Raise a DN to a power. Exponent must be zero or positive.'''

        # We're only handling (positive) integer powers...
        if isinstance(other,int):
            # Not handling negative exponents, b/c no inverse...
            if other < 0:
                raise ValueError(f'Exponent must be postive, not {other}.')
            # Handle exponent=0, return 1...
            if other == 0:
                return dualnumber(1)
            # Create a copy...
            tmp = +self
            # Handle exponent=1, return DN...
            if other == 1:
                return tmp
            # Handle higher powers...
            for _ in range(other-1):
                tmp *= self
            # Return value...
            return tmp

        # Nope...
        raise ValueError(f'Exponent must be int, not {typename(other)}.')

    def __eq__ (self, other):
        '''See if other is equal to self.'''
        if other is None: return False

        # If other is a dualnumber, compare parts...
        if isinstance(other,dualnumber):
            if self.real != other.real: return False
            if self.dual != other.dual: return False
            return True

        # If other is a scalar, compare as dualnumber(x,0)...
        if isinstance(other,int) or isinstance(other,float):
            return self == dualnumber(other)

        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __ne__ (self, other):
        '''Not-equals is just the opposite of equals.'''
        return not (self == other)

    def __bool__ (self):
        '''Return True of DN is non-zero.'''
        return False if ((self.real==0) and (self.dual==0)) else True

    def __neg__ (self):
        '''Return a negative version of self.'''
        return dualnumber(-self.real, -self.dual)

    def __pos__ (self):
        '''Return a positive version of self (just duplicates).'''
        return dualnumber(self.real, self.dual)

    def __abs__ (self):
        '''Return the absolute value of self.'''
        return dualnumber(abs(self.real), abs(self.dual))

    def __int__ (self):
        '''Return real value as integer.'''
        return int(self.real)

    def __float__ (self):
        '''Return real value as a float.'''
        return float(self.real)


def make_chart (xs, ys0,ys1, **kwargs):
    '''Create a chart.'''
    from os import path
    from matplotlib import pyplot as plt
    from matplotlib.ticker import MultipleLocator
    from matplotlib.ticker import FormatStrFormatter
    from matplotlib.patches import Patch

    fname  = kwarg('fname', 'dualnums.png', str, kwargs)
    fpath  = kwarg('fpath', r'C:\Demo\HCC\Python', str, kwargs)
    xmin   = kwarg('xmin', -5, int, kwargs)
    xmax   = kwarg('xmax', +5, int, kwargs)
    ymin   = kwarg('ymin', -10, int, kwargs)
    ymax   = kwarg('ymax', +10, int, kwargs)
    xticks = kwarg('xticks', (1.0, 0.1), tuple, kwargs)
    yticks = kwarg('yticks', (1.0, 0.1), tuple, kwargs)
    fcolor = kwarg('fc', '#7f7f7f', str, kwargs)
    gcolor = kwarg('gc', '#ff0000', str, kwargs)
    latex = r'$\left[=\frac{d}{dx}f(x)\right]$'
    title  = f'f(x) & g(x) {latex} with dual numbers'

    # Create a new plot 6.0 inches wide and 8.0 inches tall...
    fig, ax = plt.subplots()
    fig.set_figwidth(6.0)
    fig.set_figheight(8.0)
    fig.suptitle(title, fontsize=14, fontweight='bold')
    fig.subplots_adjust(left=0.065, right=0.990, bottom=0.07, top=0.93)

    # Define X-axis properties...
    ax.set_xlim(xmin, xmax)
    ax.xaxis.set_major_locator(MultipleLocator(xticks[0]))
    ax.xaxis.set_minor_locator(MultipleLocator(xticks[1]))
    ax.xaxis.set_major_formatter(FormatStrFormatter('%.0f'))

    # Define Y-axis properties...
    ax.set_ylim(ymin, ymax)
    ax.yaxis.set_major_locator(MultipleLocator(yticks[0]))
    ax.yaxis.set_minor_locator(MultipleLocator(yticks[1]))
    ax.yaxis.set_major_formatter(FormatStrFormatter('%.0f'))

    # Define background grid properties...
    props = dict(axis='both', ls='-', alpha=1.0)
    ax.grid(which='major', color='#9f9f9f', lw=1.0, zorder=1, **props)
    ax.grid(which='minor', color='#bfbfbf', lw=0.6, zorder=0, **props)

    props = dict(axis='both', direction='out', color='#000000')
    ax.tick_params(which='major', **props)
    ax.tick_params(which='minor', **props)

    # Draw the X and Y axes...
    ax.axhline(0.0, lw=1.2, color='#000000', alpha=1.00, zorder=10)
    ax.axvline(0.0, lw=1.2, color='#000000', alpha=1.00, zorder=10)

    # Plot f(x) and g(x)...
    ax.plot(xs,ys0, label='f(x)', c=fcolor, lw=2.5, alpha=0.8, zorder=11)
    ax.plot(xs,ys1, label='g(x)', c=gcolor, lw=2.5, alpha=0.8, zorder=12)

    # Define legend...
    patchs = [
        Patch(facecolor=fcolor, edgecolor='#000000', lw=0.75),
        Patch(facecolor=gcolor, edgecolor='#000000', lw=0.75),
    ]
    labels = ['f(x)', 'g(x)']
    fig.legend(
        patchs,
        labels,
        loc='lower right',
        ncols=2,
        fontsize=10,
        bbox_to_anchor=(0.99,0.00)
    )

    # Save the chart as an image file...
    filename = path.join(fpath,fname) if fpath else fname
    fig.savefig(filename)
    print(f'wrote: {filename}')
    print()



'''eof'''
