typename = lambda obj: type(obj).__name__

class dualnumber:
    Etype = 'Sorry. Unknown type: {}'

    def __init__ (self, real=0, dual=0):
        '''New Dual Number instance.'''
        self.real = real
        self.dual = dual

    def __repr__ (self):
        '''Return debug string.'''
        return f'<{typename(self)} @{id(self):08x}>'

    def __str__ (self):
        '''Return printable string.'''
        return f'({self.real:+}{self.dual:+}e)'

    @staticmethod
    def differentiate (function, x):
        '''Differentiate function at x and return value.'''
        return function(dualnumber(x,1)).dual

    def __add__  (self, other):
        '''Add with a DN or a numeric value and return new DN.'''

        # First see if other is another dualnumber...
        if isinstance(other,dualnumber):
            # Do the math...
            ta = self.real + other.real
            tb = self.dual + other.dual
            # Return a new dual number...
            return dualnumber(ta, tb)

        # Alternately, handle integers and floats...
        if isinstance(other,int) or isinstance(other,float):
            # The math here involves just the real part...
            ta = self.real + other
            tb = self.dual
            # Return a new dual number...
            return dualnumber(ta, tb)

        # If none of the above, that's an error...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    # Addition is communitive, so Right-Add is same as Left-Add...
    __radd__ = __add__

    def __iadd__  (self, other):
        '''Add a DN or numeric value to self.'''

        # First, deal with another dualnumber...
        if isinstance(other,dualnumber):
            # Do the math to self...
            self.real += other.real
            self.dual += other.dual
            return self

        # Alternately, integers or floats...
        if isinstance(other,int) or isinstance(other,float):
            self.real += other
            return self

        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __sub__  (self, other):
        '''Subtract with a DN or a numeric value and return new DN.'''
        if isinstance(other,dualnumber):
            ta = self.real - other.real
            tb = self.dual - other.dual
            return dualnumber(ta, tb)
        if isinstance(other,int) or isinstance(other,float):
            ta = self.real - other
            tb = self.dual
            return dualnumber(ta, tb)
        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __rsub__ (self, other):
        '''Subtract with a DN or a numeric value and return new DN.
           Note that subtraction does NOT commute so we must implement rsub.
           Note also that other will never be a dualnumber, so we only need
           to handle ints and floats.'''
        if isinstance(other,int) or isinstance(other,float):
            ta = other - self.real
            tb = self.dual
            return dualnumber(ta, tb)
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __isub__  (self, other):
        '''Subtract a DN or numeric value from self.'''
        if isinstance(other,dualnumber):
            self.real -= other.real
            self.dual -= other.dual
            return self
        if isinstance(other,int) or isinstance(other,float):
            self.real -= other
            return self
        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __mul__  (self, other):
        '''Multiply by a DN or a numeric value and return new DN.'''

        # The multiplication math is a bit more involved...
        if isinstance(other,dualnumber):
            ta = self.real * other.real
            tb = (self.real * other.dual) + (self.dual * other.real)
            return dualnumber(ta, tb)

        # Scalar values apply to both real and dual parts...
        if isinstance(other,int) or isinstance(other,float):
            ta = self.real * other
            tb = self.dual * other
            return dualnumber(ta, tb)
        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    # Right-Mul same as Left-Mul...
    __rmul__ = __mul__

    def __imul__  (self, other):
        '''Multiply a DN or numeric value to self.'''
        if isinstance(other,dualnumber):
            ta = self.real * other.real
            tb = (self.real * other.dual) + (self.dual * other.real)
            self.real = ta
            self.dual = tb
            return self
        if isinstance(other,int) or isinstance(other,float):
            self.real *= other
            self.dual *= other
            return self
        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __pow__ (self, other, modulo=None):
        '''Raise a DN to a power. Exponent must be zero or positive.'''

        # We're only handling (positive) integer powers...
        if isinstance(other,int):
            # Not handling negative exponents, b/c no inverse...
            if other < 0:
                raise ValueError(f'Exponent must be postive, not {other}.')
            # Handle exponent=0, return 1...
            if other == 0:
                return dualnumber(1)
            # Create a copy...
            tmp = +self
            # Handle exponent=1, return DN...
            if other == 1:
                return tmp
            # Handle higher powers...
            for _ in range(other-1):
                tmp *= self
            # Return value...
            return tmp

        # Nope...
        raise ValueError(f'Exponent must be int, not {typename(other)}.')

    def __eq__ (self, other):
        '''See if other is equal to self.'''
        if other is None: return False

        # If other is a dualnumber, compare parts...
        if isinstance(other,dualnumber):
            if self.real != other.real: return False
            if self.dual != other.dual: return False
            return True

        # If other is a scalar, compare as dualnumber(x,0)...
        if isinstance(other,int) or isinstance(other,float):
            return self == dualnumber(other)

        # Nope...
        raise ValueError(dualnumber.Etype.format(typename(other)))

    def __ne__ (self, other):
        '''Not-equals is just the opposite of equals.'''
        return not (self == other)

    def __bool__ (self):
        '''Return True of DN is non-zero.'''
        return False if ((self.real==0) and (self.dual==0)) else True

    def __neg__ (self):
        '''Return a negative version of self.'''
        return dualnumber(-self.real, -self.dual)

    def __pos__ (self):
        '''Return a positive version of self (just duplicates).'''
        return dualnumber(self.real, self.dual)

    def __abs__ (self):
        '''Return the absolute value of self.'''
        return dualnumber(abs(self.real), abs(self.dual))

    def __int__ (self):
        '''Return real value as integer.'''
        return int(self.real)

    def __float__ (self):
        '''Return real value as a float.'''
        return float(self.real)

