'''Examples.py'''
from time import perf_counter_ns


def DataRecorder (which):
    '''Decorator for logging a specific parameter of function calls.'''
    print(f'DataRecorder({which})')
    if not (isinstance(which,int) or isinstance(which,str)):
        raise TypeError(f'Parameter spec must be int or string.')

    def decorator (function):
        '''Actual decorator function returned in initialization.'''
        ftn_name = function.__qualname__
        print(f'@DataRecorder: {ftn_name}')

        def wrapper (*args, **kwargs):
            '''Wrapper function.'''
            wrapper.count += 1
            print(f'{ftn_name}[{wrapper.count}]: {args} {kwargs}')

            # An integer indexes a parameter in args...
            if isinstance(which,int) and (which < len(args)):
                arg = args[which]
                wrapper.data.append((wrapper.count, arg))

            # A string names a parameter in kwargs...
            elif isinstance(which,str) and (which in kwargs):
                arg = kwargs[which]
                wrapper.data.append((wrapper.count, arg))

            # Call wrapped function and return its result...
            return function(*args, **kwargs)

        # Initialize the wrapper object's special properties...
        wrapper.count = 0
        wrapper.data = []

        # Return the function wrapper...
        return wrapper

    # Return the decorator...
    return decorator


def StringHandler (arg_indexes, kw_names, ftn_list):
    '''Decorator for string handling; selectable args and functions.'''

    def decorator (function):
        '''Actual decorator function.'''
        ftn_name = function.__qualname__
        print(f'@StringHandler: {ftn_name}')

        def wrapper (*args, **kwargs):
            '''Wrapper function.'''
            #print(f'StringHandler: {args} {kwargs}')
            ftns = list(ftn_list) # (need mutable copies)
            _args = list(args)

            # Process positional arguments...
            for ax in arg_indexes:
                s = _args[ax]
                f = ftns.pop(0)
                _args[ax] = f(s)

            # Process keyword arguments...
            for kw in kw_names:
                s = kwargs[kw]
                f = ftns.pop(0)
                kwargs[kw] = f(s)

            # Call the function with modified args...
            retv = function(*tuple(_args), **kwargs)

            # (Optional) Process the return argument...
            if 0 < len(ftns):
                f = ftns.pop(0)
                retv = f(retv)

            # Return the function's return value...
            return retv

        # Return the function wrapper...
        return wrapper

    # Return the decorator...
    return decorator


def Byte2String (which=1, encoding='utf8'):
    '''Decorator for converting bytes to string.'''
    print(f'Byte2String({which}, {encoding})')

    # We expect 1,2,3, for argument indexes, 0 for return value...
    ix = which - 1

    def decorator (function):
        '''Decorator function.'''
        ftn_name = function.__qualname__
        print(f'@Byte2String: {ftn_name}')

        def wrapper (*args):
            '''Wrapper function.'''
            _args = list(args)

            # Positive or zero index for arguments...
            if 0 <= ix:
                # Index and convert a passed argument...
                _args[ix] = str(_args[ix], encoding=encoding)

            # Call function...
            retv = function(*_args)

            # Negative index for return value...
            if ix < 0:
                # Convert return argument...
                return str(retv, encoding=encoding)

            # Return value...
            return retv

        # Return function wrapper...
        return wrapper

    # Return decorator function...
    return decorator

def String2Byte (which=1, encoding='utf8'):
    '''Decorator for converting string to bytes.'''
    print(f'String2Byte({which}, {encoding})')

    # We expect 1,2,3, for argument indexes, 0 for return value...
    ix = which - 1

    def decorator (function):
        '''Decorator function.'''
        ftn_name = function.__qualname__
        print(f'@String2Byte: {ftn_name}')

        def wrapper (*args):
            '''Wrapper function.'''
            _args = list(args)

            # Positive or zero index for arguments...
            if 0 <= ix:
                # Index and convert a passed argument...
                _args[ix] = bytes(_args[ix], encoding=encoding)

            # Call function...
            retv = function(*_args)

            # Negative index for return value...
            if ix < 0:
                # Convert return argument...
                return bytes(retv, encoding=encoding)

            # Return value...
            return retv

        # Return function wrapper...
        return wrapper

    # Return decorator function...
    return decorator


FunctionTimes = {'main':{}}

def FunctionTimer (group='main'):
    '''Decorator function for timing functions.'''
    print(f'@FunctionTimer({group})')

    # Add requested group if it doesn't exist...
    if group not in FunctionTimes:
        FunctionTimes[group] = {}

    def decorator (function):
        '''The actual decorator function.'''
        func_name = function.__qualname__
        print(f'@FunctionTimer.decorator({func_name})')

        if func_name not in FunctionTimes[group]:
            FunctionTimes[group][func_name] = [0.0]

        def wrapper (*args, **kwargs):
            '''The wrapper function.'''
            print(f'FunctionTimer[{func_name}]')

            # Call the function and capture start and end NS...
            t0 = perf_counter_ns()
            retv = function(*args, **kwargs)
            t1 = perf_counter_ns() - t0

            # Log the total time and this call time...
            FunctionTimes[group][func_name][0] += t1
            FunctionTimes[group][func_name].append(t1)
            print(f'[{func_name}]: {t1/pow(10,6)} msecs')

            # Return timed function's value (if any)...
            return retv

        # Return the wrapper function...
        return wrapper

    # Return the actual decorator function...
    return decorator

FunctionTimer.data = FunctionTimes


def AddSimpleReprMethod (cls):
    '''Decorator function to add repr() function to a class.'''
    print(f'@AddSimpleReprMethod({cls.__name__})')

    def my_repr (self):
        return f"<{type(self).__name__} @{id(self):08x}>"

    cls.__repr__ = my_repr
    return cls

def AddReprMethod (cls):
    '''Decorator function to modify repr() function of a class.'''
    print(f'Change repr for {cls.__name__}')

    def my_repr (self):
        attrs = (f"{key}={val!r}" for key,val in vars(self).items())
        return f"{type(self).__name__}({', '.join(attrs)})"

    cls.__repr__ = my_repr
    return cls


def StaticMethod (method):
    '''Decorator to turn a method into a static method.'''

    # Gather some basic info...
    ftn_name = method.__name__
    print(f'@StaticMethod: {ftn_name}')

    class StaticMethodProperty:
        '''Property class to implement the static method wrapper.'''

        def __init__ (self, ftn):
            print(f'{ObjIdStr(self)} init')
            self.method = ftn

        def __set_name__ (self, owner, name):
            # Remember the function's name...
            self.name = name

        def __get__ (self, obj, cls):
            '''Access the property.'''
            caller = '<null>' if obj is None else ObjIdStr(obj)
            print(f'{ObjIdStr(self)} get: {caller}')
            def proxy (*args, **kwargs):
                '''Proxy function that wraps class method.'''
                return self.method(*args, **kwargs)
            return proxy

        def __set__ (self, obj, value):
            '''(Only protects calls on instance objects.'''
            raise NotImplementedError(f'You may not modify {self.name}!')

        def __delete__ (self, obj):
            '''(Only protects calls on instance objects.'''
            raise NotImplementedError(f'You may not delete {self.name}!')

    # Return new instance of StaticMethodProperty...
    return StaticMethodProperty(method)

ObjIdStr = lambda obj: f'<{type(obj).__name__} @{id(obj):08x}>'

def ClassMethod (method):
    '''Decorator to turn a method into a class method.'''

    ftn_name = method.__name__
    print(f'@ClassMethod: {ftn_name}')

    class ClassMethodProperty:
        '''Property class to implement the class method wrapper.'''

        def __init__ (self, ftn):
            #print(f'{ObjIdStr(self)} init')
            self.method = ftn

        def __set_name__ (self, owner, name):
            # Remember the function's name...
            self.name = name

        def __get__ (self, obj, cls):
            '''Access the property.'''
            caller = '<null>' if obj is None else ObjIdStr(obj)
            print(f'{ObjIdStr(self)} get: {caller}')
            def proxy (*args, **kwargs):
                '''Proxy function that wraps class method.'''
                return self.method(cls, *args, **kwargs)
            return proxy

        def __set__ (self, obj, value):
            '''(Only protects calls on instance objects.'''
            raise NotImplementedError(f'You may not modify {self.name}!')

        def __delete__ (self, obj):
            '''(Only protects calls on instance objects.'''
            raise NotImplementedError(f'You may not delete {self.name}!')

    # Return new instance of ClassMethodProperty...
    return ClassMethodProperty(method)


class BasicDecoratorClass:
    '''Basic Decorator *class*. Use instances for decorators.'''

    class DecoratorFunctionClass:
        '''Decorator class for wrapping a function.'''

        def __init__ (self, function):
            '''New wrapper instance; receives decorated function.'''
            print(f'@DecoratorFunction({function})')
            self.function = function

        def __call__ (self, *args, **kwargs):
            '''Call to the decorated function.'''
            print(f'Decorator({len(args)} args, {len(kwargs)} kwargs)')

            # Call the wrapped function; return its return value...
            return self.function(*args, **kwargs)

    def __init__ (self):
        '''New decorator instance.'''
        print(f'@Decorator()')

    def __call__ (self, function):
        '''Return a new wrapper for a function.'''
        return BasicDecoratorClass.DecoratorFunctionClass(function)



'''eof'''
