from examples import MimeHeader

EOL  = ord('\n')
CR   = ord('\r')
DASH = ord('-')

def parse_multipart_form (content_type:str, byte_string:bytes) -> tuple:
    '''\
Parse Multipart Form Data.

Arguments:
    content_type    -content header text
    byte_string     -form data bytes

Returns:
    tuple: (form, filename, filetype, filedata)

    Note: the last three field will be None if no file data.
'''
    form = {}               # form data to be returned
    parthdrs = {}           # temp dictionary for part headers
    content = []            # buffer for building file content
    buf = []                # temp buff for building strings
    state = 'prefix'        # current state
    boundary = None         # part separator string

    # Parse the content-type header to get the multipart boundary...
    ix = content_type.index('boundary=') + len('boundary=')
    boundary = content_type[ix:].lstrip('-')

    # Iterate over the bytes of the data...
    for bx in byte_string:

        if state == 'prefix':
            # Accumulate dashes...
            if bx == DASH:
                buf.append(bx)
                continue
            # Not a dash; separator text begins...
            if len(buf) < 6:
                raise SyntaxError(f'Expected six hyphens, not {len(buf)}.')
            # Add first separator character to buffer...
            buf = [bx]
            state = 'sep'
            continue

        if state == 'sep':
            # End of separator...
            if bx == EOL:
                sep_str = ''.join(chr(b) for b in buf)
                if not sep_str.startswith(boundary):
                    raise SyntaxError(f'Invalid Separator: "{sep_str}"')
                buf = []
                parthdrs = {}
                state = 'part.start'
                continue
            # Ignore CR characters...
            if bx == CR:
                continue
            # Add separator character to buffer...
            buf.append(bx)
            continue

        if state == 'part.start':
            # Blank line (instead of header text)...
            if bx == EOL:
                # End of part headers...
                content = []
                state = 'part.data'
                continue
            # Ignore CR characters...
            if bx == CR:
                continue
            # Add first part header character to buffer...
            buf = [bx]
            state = 'part.hdr'
            continue

        if state == 'part.hdr':
            # Handle end of line...
            if bx == EOL:
                txt = ''.join(chr(b) for b in buf)
                hdr = MimeHeader(txt)
                parthdrs[hdr.name.lower()] = hdr
                state = 'part.start'
                continue
            # Ignore CR characters...
            if bx == CR:
                continue
            # Add part header character to buffer...
            buf.append(bx)
            continue

        if state == 'part.data':
            # A dash might mean a separator...
            if bx == DASH:
                buf = [bx]
                state = 'test.prefix'
                continue
            # Add content byte to buffer...
            content.append(bx)
            continue

        if state == 'test.prefix':
            # Not a dash; start of separator...
            if bx != DASH:
                # Not a separator; add to content and resume...
                if len(buf) != 6:
                    # Not a separator...
                    content.extend(buf)
                    content.append(bx)
                    state = 'part.data'
                    continue
                # Got 6 dashes; might be a separator...
                buf = [bx]
                state = 'test.sep'
                continue
            # Add character to possible prefix string...
            buf.append(bx)
            continue

        if state == 'test.sep':
            # End of line; test separator...
            if bx == EOL:
                txt = (''.join(chr(c) for c in buf)).strip()
                if txt != boundary:
                    # Not a separator...
                    content.extend(buf)
                    content.append(bx)
                    state = 'part.data'
                    continue
                # It is a separator; get disposition header...
                cdisp = parthdrs['content-disposition']
                name = cdisp.fields['name']
                # Add file content (bytes) to form (strip trailing CR-LF)...
                form[name] = bytes(content[0:-2])
                # Add the disposition header...
                form[f'{name}.disp'] = cdisp
                # If content-type header provided...
                if 'content-type' in parthdrs:
                    # Add to form...
                    form[f'{name}.type'] = parthdrs['content-type']
                # Get ready for next part...
                buf = []
                parthdrs = {}
                state = 'part.start'
                continue
            # Add to separator string...
            buf.append(bx)
            continue

    # Extract the filename, filetype, and file data for convenience...
    filedata, filename, filetype = None,None,None
    for name in form:
        if name == 'fn':
            filedata = form['fn']
            filetype = form['fn.type'].text
            filename = form['fn.disp'].fields['filename']
            if filetype.startswith('text'):
                filedata = str(filedata, encoding='utf8')
            continue

        # Also convert simple fields from bytes to strings...
        if '.' not in name:
            txt = form[name]
            if isinstance(txt,bytes):
                form[name] = str(txt, encoding='utf8')

    form['fn.name'] = filename
    return (form, filename, filetype, filedata)

