txt = 'Content-Disposition: form-data; name="fn"; filename="foo.py"'

class MimeHeader (tuple):
    '''\
MIME Header class. Parses text to a defined tuple.

Expected text has the syntax:
    <name>: <text> [; name=value [; ...]]

Created tuple is:
    (name, text, fields)

Properties:
    name        -field name
    text        -field text
    fields      -dictionary of name:value pairs

Methods:
    str(obj)    -printable version
    repr(obj)   -debugging version
'''
    def __new__ (cls, text):
        '''New MimeHeader instance.'''

        # Split text into name and value parts...
        parts = text.partition(': ')
        if parts[1] != ': ':
            ValueError(f'Invalid MIME Header (no ": "): "{text}"')

        # Split the value into fields...
        subparts = parts[2].split('; ')
        value = subparts[0]

        fields = {}
        for subp in subparts[1:]:
            # Split field into name and value...
            key,val = subp.split('=', maxsplit=1)
            # Add field; remove any surrounding double-quotes...
            fields[key] = val.strip('"')

        # Delegate creating new instance to tuple...
        return super().__new__(cls, (parts[0], value, fields))

    @property
    def name (self): return self[0]

    @property
    def text (self): return self[1]

    @property
    def fields (self): return self[2]

    def __str__ (self):
        return f'{self.name}: {self.text}'

    def __repr__ (self):
        return f'<{type(self).__name__} @{id(self):08x}>'


if __name__ == '__main__':
    hdr = MimeHeader(txt)
    print(f'Header: {hdr} {hdr!r}')
    print(f'Header Name: {hdr.name}')
    print(f'Header Text: {hdr.text}')
    print('Header Fields:')
    for nam,val in hdr.fields.items():
        print(f'> {nam}: {val}')
    print()

