'''Examples.py'''


def fib_seq_print (seq):
    '''Print a Fibonacci sequence.'''

    for ix,fn in enumerate(seq):
        print(f'{ix+1:4d} = {fn}')
    print()


def print_timing (name, seq_len, ns):
    '''Print function timing.'''
    s1 = f'{name}[{seq_len}]'
    print(f'{s1:<28}: {ns/pow(10,6):10.3f} ms')


def basic_fib_seq (seq_len):
    '''Basic Fibonacci sequence generator.'''

    def fib (n):
        '''Canonical Recursive Fibonacci function.'''
        if n in [0,1]: return 1
        return (fib(n-2) + fib(n-1))

    return [fib(n) for n in range(seq_len)]


def cache (obj):
    '''Cache Decorator for fib function.'''
    cache.previous = {}

    def wrapper (n):
        if n in cache.previous:
            return cache.previous[n]
        r = obj(n)
        cache.previous[n] = r
        return r

    return wrapper


def cached_fib_seq (seq_len):
    '''Cached Fibonacci sequence generator.'''

    @cache
    def fib (n):
        '''Canonical Recursive Fibonacci function.'''
        if n in [0,1]: return 1
        return (fib(n-2) + fib(n-1))

    return [fib(n) for n in range(seq_len)]


def array_fib_seq (seq_len):
    '''Array cache Fibonacci sequence generator.'''

    seq = [0]*seq_len
    seq[0] = 1
    seq[1] = 1
    seq[2] = 2

    def fib (n):
        if seq[n]: return seq[n]
        seq[n] = (fib(n-2) + fib(n-1))
        return 

    for n in range(seq_len):
        fib(n)

    return seq


def iterative_fib_seq (seq_len):
    '''Iterative Fibonacci sequence generator.'''

    seq = [0]*seq_len
    seq[0] = 1
    seq[1] = 1
    seq[2] = 2

    def fib (n):
        if seq[n]: return seq[n]
        seq[n]
        return 

    for n in range(3, seq_len):
        seq[n] = seq[n-2] + seq[n-1]

    return seq


def mechanical_fib_seq (seq_len, left_hand=1, right_hand=1):
    '''Mechanical Iterative Fibonacci sequence generator.'''

    # Start with something(s) in left and right hands...
    left = left_hand
    right = right_hand
    # These are the first two Fibonacci numbers...
    seq = [left, right]

    # Generate sequence with simple process...
    for ix in range(seq_len-2):
        # Copy left and right hands to pile...
        pile = left + right
        # The pile is the next Fibonacci number...
        seq.append(pile)
        # Move right hand stuff to left hand...
        # (Left hand stuff is discarded.)
        left = right
        # Move the pile to the right hand...
        right = pile
        # Rinse and repeat...

    # Return the Fibonacci sequence...
    return seq



'''eof'''
