'''Examples.py'''
import hashlib

bf_Sizes = [16, 32, 64, 128, 256]

def bf_charset (charset, size=32):
    '''Bloom filter for a small set of chars.'''
    assert size in bf_Sizes, ValueError(f'Invalid size: {size}.')

    # Create mask for indexing vector...
    mask = size-1

    # Create vector of specified size...
    vector = [0]*size

    # Add each char in charset to filter...
    for ch in charset:
        # Use ordinal value as index...
        cx = ord(ch) & mask
        # Set vector "bit" to one...
        vector[cx] = 1

    # Return the filter vector...
    return vector

def bf_char_check (bf, char, size=32):
    '''Test for charset Bloom filter.'''
    assert size in bf_Sizes, ValueError(f'Invalid size: {size}.')

    # Create mask for indexing vector...
    mask = size-1

    # Use ordinal value as index...
    cx = ord(char) & mask

    # Return filter bit...
    return bf[cx]



class bf_base:
    '''Bloom Filter Base class.'''

    def __init__(self, size):
        '''New BloomFilterBase instance.'''

        # Size of the "bit" vector...
        self.size = size

        # Table of hash function constructors...
        self.hash_functions = [hashlib.md5,
                               hashlib.sha1,
                               hashlib.sha384,
                               hashlib.sha256,
                               hashlib.sha512]

        # Size of the hash function table...
        self.nhash = len(self.hash_functions)


    def _hash(self, data:str, which:int) -> int:
        '''Return a hash of data.'''

        # Get a valid index into the table of hashes...
        wx = which % len(self.hash_functions)

        # Get hash function constructor from the table...
        hf = self.hash_functions[wx]

        # Get hash function...
        hasher = hf()

        # Send data through the hasher...
        hasher.update(data.encode('utf-8'))

        # Get the digest as hex string and convert to int...
        digest = int(hasher.hexdigest(), 16)

        # Return lower N bits of hash number...
        return digest % self.size


class bf_basic (bf_base):
    '''Basic Bloom Filter.'''

    def __init__ (self, size):
        '''New BloomFilter instance.'''
        super().__init__(size)

        # Number of bytes required to store size number of elements
        self.nbytes = (size + 7) // 8 

        # Initialize a bytearry with all zeros
        self.bit_vector = bytearray(([0] * self.nbytes))

    def query (self, item) -> bool:
        '''Test filter to see if item exists.'''

        # For each hash function in the table...
        for ix in range(self.nhash):

            # Get a hashed index for the item...
            index = self._hash(item, ix)

            # Get the byte and bit indexes...
            byte_index, bit_index = divmod(index, 8)

            # Create a bit mask for the bit index...
            mask = 1 << bit_index

            # If the matching bit in the vector is 0...
            if (self.bit_vector[byte_index] & mask) == 0:
                # Item is not in the filter...
                return False

        # If none of the bits were 0, item MAY be in the filter...
        return True

    def add (self, item) -> None:
        '''Add an item to the filter.'''

        # For each hash function in the table...
        for ix in range(self.nhash):

            # Get a hashed index for the item...
            index = self._hash(item, ix)

            # Get the byte and bit indexes...
            byte_index, bit_index = divmod(index, 8)

            # Create a bit mask for the bit index...
            mask = 1 << bit_index

            # Set the indexe bit in the vector...
            self.bit_vector[byte_index] |= mask

    def blanks (self) -> int:
        '''Return number of unused vector slots.'''
        func = lambda b:sum(0 if b & pow(2,bx) else 1 for bx in range(8))
        return sum(func(b) for b in self.bit_vector)

    def __len__ (self) -> int: return self.bit_vector
    def __getitem__ (self, ix) -> bytes: return self.bit_vector[ix]
    def __iter__ (self): return iter(self.bit_vector)

    def __str__ (self) -> str:
        return self._vector2bits()

    def _byte2bits (self, b) -> str:
        '''Convert a byte to a string of '1' and '0'.'''
        bs = [('1' if b & pow(2,bx) else '0') for bx in range(8)]
        return ''.join(reversed(bs))

    def _vector2bits (self) -> str:
        '''Convert vector into a string of bits.'''
        bs = [self._byte2bits(b) for b in reversed(self.bit_vector)]
        return '.'.join(bs)


class bf_counting (bf_base):
    ...



class BloomFilterBase (bf_base):
    ...



'''eof'''
