from examples import BloomFilterBase

class BloomFilterCounting (BloomFilterBase):
    '''Counting Bloom Filter.'''

    def __init__ (self, size):
        '''New BloomFilterCounting instance.'''
        super().__init__(size)

        # Initialize number vector with all zeros
        self.nbr_vector = [0]*self.size

    def query (self, item:str) -> bool:
        '''Test filter to see if item exists.'''

        # For each hash function in the table...
        for ix in range(self.nhash):

            # Get a hashed index for the item...
            index = self._hash(item, ix)

            # If slot is zero, object isn't in filter...
            if self.nbr_vector[index] == 0:
                return False

        # If none of the slots was 0, item MAY be in the filter...
        return True

    def add (self, item) -> None:
        '''Add an item to the filter.'''

        # For each hash function in the table...
        for ix in range(self.nhash):

            # Get a hashed index for the item...
            index = self._hash(item, ix)

            # Increment the indexed slot...
            self.nbr_vector[index] += 1

    def delete (self, item) -> None:
        '''Remove an item from the filter.'''

        # If the item isn't in the filter, raise an error...
        if not self.query(item):
            raise RuntimeError('Item Not Found.')

        # For each hash function in the table...
        for ix in range(self.nhash):

            # Get a hashed index for the item...
            index = self._hash(item, ix)

            # Increment the indexed slot...
            self.nbr_vector[index] -= 1

            # This should never happen...
            if self.nbr_vector[index] < 0:
                raise RuntimeError('Counter Underflow.')

    def blanks (self) -> int:
        '''Return number of unused vector slots.'''
        return int(sum(1 if n==0 else 0 for n in self.nbr_vector))

    def __len__ (self) -> int: return self.nbr_vector
    def __getitem__ (self, ix) -> bytes: return self.nbr_vector[ix]
    def __iter__ (self): return iter(self.nbr_vector)

    def __str__ (self) -> str:
        return ''.join([str(n) for n in self.nbr_vector])

bf = BloomFilterCounting(64)
for w in ['who', 'what', 'why', 'where', 'when']:
    bf.add(w)

print(f'{bf=!s}')
print(f'> blanks={bf.blanks()}')
print()
print(f'test ""       : {bf.query("")}')
print(f'test "when"   : {bf.query("when")}')
print(f'test "went"   : {bf.query("went")}')
print(f'test "why"    : {bf.query("why")}')
print(f'test "why not": {bf.query("why not")}')
print(f'test "where"  : {bf.query("where")}')
print(f'test "who"    : {bf.query("who")}')
print(f'test "wh"     : {bf.query("wh")}')
print(f'test "am"     : {bf.query("am")}')
print()
print('delete("where")')
bf.delete('where')
print(f'test "where"  : {bf.query("where")}')
print()

