from examples import BloomFilterBase

class BloomFilter (BloomFilterBase):
    '''Basic Bloom Filter.'''

    def __init__ (self, size):
        '''New BloomFilter instance.'''
        super().__init__(size)

        # Number of bytes required to store size number of elements
        self.nbytes = (size + 7) // 8

        # Initialize a bytearry with all zeros
        self.bit_vector = bytearray(([0] * self.nbytes))

    def query (self, item) -> bool:
        '''Test filter to see if item exists.'''

        # For each hash function in the table...
        for ix in range(self.nhash):

            # Get a hashed index for the item...
            index = self._hash(item, ix)

            # Get the byte and bit indexes...
            byte_index, bit_index = divmod(index, 8)

            # Create a bit mask for the bit index...
            mask = 1 << bit_index

            # If the matching bit in the vector is 0...
            if (self.bit_vector[byte_index] & mask) == 0:
                # Item is not in the filter...
                return False

        # If none of the bits were 0, item MAY be in the filter...
        return True

    def add (self, item) -> None:
        '''Add an item to the filter.'''

        # For each hash function in the table...
        for ix in range(self.nhash):

            # Get a hashed index for the item...
            index = self._hash(item, ix)

            # Get the byte and bit indexes...
            byte_index, bit_index = divmod(index, 8)

            # Create a bit mask for the bit index...
            mask = 1 << bit_index

            # Set the indexe bit in the vector...
            self.bit_vector[byte_index] |= mask

    def blanks (self) -> int:
        '''Return number of unused vector slots.'''
        func = lambda b:sum(0 if b & pow(2,bx) else 1 for bx in range(8))
        return sum(func(b) for b in self.bit_vector)

    def __len__ (self) -> int: return self.bit_vector
    def __getitem__ (self, ix) -> bytes: return self.bit_vector[ix]
    def __iter__ (self): return iter(self.bit_vector)

    def __str__ (self) -> str:
        return self._vector2bits()

    def _byte2bits (self, b) -> str:
        '''Convert a byte to a string of '1' and '0'.'''
        bs = [('1' if b & pow(2,bx) else '0') for bx in range(8)]
        return ''.join(reversed(bs))

    def _vector2bits (self) -> str:
        '''Convert vector into a string of bits.'''
        bs = [self._byte2bits(b) for b in reversed(self.bit_vector)]
        return '.'.join(bs)


bf = BloomFilter(64)
for w in ['who', 'what', 'why', 'where', 'when']:
    bf.add(w)

print(f'{bf=!s}')
print(f'> blanks={bf.blanks()}')
print()
print(f'test ""       : {bf.query("")}')
print(f'test "when"   : {bf.query("when")}')
print(f'test "went"   : {bf.query("went")}')
print(f'test "why"    : {bf.query("why")}')
print(f'test "why not": {bf.query("why not")}')
print(f'test "where"  : {bf.query("where")}')
print(f'test "who"    : {bf.query("who")}')
print(f'test "wh"     : {bf.query("wh")}')
print(f'test "am"     : {bf.query("am")}')
print()

