'''Library classes for Simple Python Tricks #4'''
from os import path
from time import localtime
from datetime import datetime

class fileobj:
    '''File Object class.'''

    def __init__ (self, filename):
        '''New File Object instance.'''
        self.fullname = filename

        # Parse the full name into parts...
        t = path.split(self.fullname)
        self.pathname = t[0]
        self.filename = t[1]
        t = path.splitext(self.filename)
        self.name  = t[0]
        self.ext = t[1][1:]

        # Default file properties...
        self.fsize = 0
        self.created = None
        self.updated = None

        # If the file exists, get its properties...
        self.dflag = False
        self.fflag = False
        self.xflag = path.exists(self.fullname)
        if self.xflag:
            # Exists!...
            self.dflag = path.isdir(self.fullname)
            self.fflag = path.isfile(self.fullname)

            # And it's a file!...
            if self.fflag:
                # File Size...
                self.fsize = path.getsize(self.fullname)
                # File Create Date/Time...
                t = localtime(path.getctime(self.fullname))
                self.created = datetime(*t[0:6])
                # File Updated Date/Time...
                t = localtime(path.getmtime(self.fullname))
                self.updated = datetime(*t[0:6])

    def __getattribute__ (self, name):
        '''Enable virtual attributes.'''
        if name == 'exists': return self.xflag
        if name == 'isfile': return self.fflag
        if name == 'isdir':  return self.dflag
        return super().__getattribute__(name)

    def __bool__ (self):
        '''Boolean value: True if file exists.'''
        return True if self.xflag else False

    def __eq__ (self, other): return (self.fullname == other.fullname)
    def __lt__ (self, other): return (self.fullname < other.fullname)
    def __gt__ (self, other): return (other < self)
    def __le__ (self, other): return not (self < other)
    def __ge__ (self, other): return not (other < self)
    def __ne__ (self, other): return not (self == other)

    def __str__ (self): return f'{self.fullname} ({self.fsize} bytes)'


class filecontent (fileobj):
    '''Abstract File base class.'''

    def __init__ (self, filename):
        '''New File Object instance.'''
        super().__init__(filename)
        self.data = None

    def __len__ (self):
        '''File Object's length is the length of its data.'''
        return len(self.data)

    def __getitem__ (self, ix):
        '''Get a datum by index. (Assumes data is indexable!)'''
        return self.data[ix]

    def __setitem__ (self, ix, value):
        '''Set a datum by index. (Assumes data is indexable!)'''
        self.data[ix] = value

    def __iter__ (self):
        '''Set a datum by index. (Assumes data is iterable!)'''
        return iter(self.data)


'''eof'''
